/* SDL LIFE1D. One-Dimensional Cellular Automata Graphics program for SDL2.
   (C) A.Millett 1992-2025. Released as free software under GPL3.

to build: ./sdlmak sdl_life1d -Os

 -> Adapt Win ver: life2d102  (29.1.2009)   
    Simple working version, just steps to next seed on space. 
 -> SDL_LIFE1D110. (4.5.2025) (10k elf)
    Imp simple param display (title bar), keyboard cmds. (rgbdntuxcjs)
 -> SDL_LIFE1D111. (4.5.2025) 
    Imp consistant/portable io_rand()
 -> SDL_LIFE1D112. (4.5.2025) 

*/

char szProg [] = "SDL_LIFE1D v1.12. (C) A.Millett 2025,Free/GPL3.";

#include <unistd.h>
#include <SDL.h>
#include <SDL_scancode.h>

typedef unsigned char BYTE;
#define FALSE 0
#define TRUE  1

SDL_Window  * hWin = NULL;		/* Main program window */
SDL_Renderer * sRender = NULL;		/* Used for drawing comands */

SDL_DisplayMode dispMode;	/* Used to find screen size */
SDL_Event sEvent;		/* Used for event loop (kbd/mouse etc */
SDL_Rect rect;

int vdux = 0;
int vduy = 0;


  /* SDL function error, show msg, abort. */
void sdl_error (char *pMsg)
{ 
     printf ("%s error: %s\n", pMsg, SDL_GetError ());
     sleep (1);
     exit (1);
}

/*---------------------------------------------------------------
  life_ life 1d rendering etc
-----------------------------------------------------------------*/

#define START_SEED 6	/* Define starting seed/picture */
#define MAXX 4096	/* Max no of cells across */
#define MAX_RULES 256
#define MAX_NCOL 15	/* Max no of cell-states/colours */

int life_seed = 1;
BYTE life_maxcol = 5;
BYTE life_method = 14;	// Sampling method
BYTE life_jump = 0;	// Jump increment in 256 pixels (changes rd patten starter, keeps same rules)


BYTE gen  [MAXX+16];	/* Old/prev row of cells */
BYTE gen2 [MAXX+16];	/* New/Next row of cells */

BYTE rules[MAX_RULES];	/* Look up index, sum of input colours give new index */


#define DEF_RED 152	/* Default colour */
#define DEF_GREEN 128
#define DEF_BLUE 56

BYTE life_red = DEF_RED;
BYTE life_green = DEF_GREEN;
BYTE life_blue = DEF_BLUE;

#define COL_STEP 8		/* Step to alter colours */

int life_scale = 2;		/* Cell size in pixels */

char szTitle [512];

	/* Simple consistant/portable 32 bit random gen */
unsigned int io_rndseed = 0;

void io_srand (unsigned int rseed)
{
    io_rndseed = rseed; 
}

unsigned int io_rand ()
{
  io_rndseed = io_rndseed * 214013L + 2531011L;
  return (io_rndseed >> 16) & 0x7fff;
}


void app_Draw ()
{
    int xpos,ypos;
    int maxx,maxy;
    int cval;

    maxx = vdux / life_scale;
    maxy = vduy / life_scale;
    if (maxx > MAXX) maxx = MAXX;
    memset (gen,0,sizeof(gen));
    memset (gen2,0,sizeof(gen2));
    
    io_srand (life_seed + START_SEED);
    for (xpos = 0; xpos < MAX_RULES; xpos ++) {
      rules [xpos] = io_rand () % life_maxcol;
    }
    io_srand (life_seed + life_jump);
    for (xpos = 0; xpos < maxx + 2; xpos ++) {
      gen [xpos] = io_rand () % life_maxcol;
    }
    rect.y = 0;
    for (ypos = 0; ypos < maxy; ypos ++) {
      rect.x = 0;
      rect.w = life_scale; 
      rect.h = life_scale;
      for (xpos = 0; xpos < maxx; xpos ++) {
        // cval = rules [gen[xpos] + gen[xpos + 1] + gen[xpos + 2]];
	cval = 0;
        if (life_method & 16) cval += gen [xpos];
        if (life_method &  8) cval += gen [xpos + 1];
        if (life_method &  4) cval += gen [xpos + 2];
        if (life_method &  2) cval += gen [xpos + 3];
        if (life_method &  1) cval += gen [xpos + 4];
	cval = rules [cval];
	gen2[xpos + 2] = cval;
        SDL_SetRenderDrawColor (sRender, cval * life_red, cval * life_green, cval * life_blue, 255);
	//SDL_RenderDrawPoint (sRender, xpos, ypos);
        SDL_RenderFillRect (sRender, &rect);      
        rect.x += life_scale;
	if (rect.x > vdux) break;
      }
      memcpy (gen,gen2,sizeof(gen));	/* Copy new row of cells to old */
      rect.y += life_scale;
      if (rect.y > vduy) break;
    }
}

  /* Handle app events */
int app_event (int delay)
{
    while (delay > 0) {
	/* Sys/Mouse/Keyboard event managment */
      while (SDL_PollEvent (&sEvent)) {
	switch (sEvent.type) {
	  case SDL_QUIT:
	    return FALSE;

          case SDL_MOUSEBUTTONDOWN:
	    life_seed ++;
	    return TRUE;

	  case SDL_KEYDOWN:

	    switch (sEvent.key.keysym.scancode) {
	      case SDL_SCANCODE_ESCAPE:
		return FALSE;

	      case SDL_SCANCODE_SPACE:		/* Next seed */
	        life_seed ++;
		return TRUE;

	      case SDL_SCANCODE_D:	/* Reset defaults */
		life_seed = 1; 
		life_maxcol = 5; 
		life_method = 14;
		life_jump = 0;
		life_red = DEF_RED; life_green = DEF_GREEN; life_blue = DEF_BLUE;
		return TRUE;

	      case SDL_SCANCODE_N:	/* No of cell states (colors) */
		life_maxcol ++;
		if (life_maxcol > MAX_NCOL) life_maxcol = 2;
		return TRUE;

	      case SDL_SCANCODE_U:
		life_seed --;
		return TRUE;

	      case SDL_SCANCODE_R:		/* RGB - Step thro colours */
		life_red = life_red + COL_STEP;
		return TRUE;

	      case SDL_SCANCODE_G:
		life_green = life_green + COL_STEP;
		return TRUE;

	      case SDL_SCANCODE_B:
		life_blue = life_blue + COL_STEP;
		return TRUE;

	      case SDL_SCANCODE_X:
		life_seed += 10;
		return TRUE;

	      case SDL_SCANCODE_T:
		life_seed -= 10;
		return TRUE;

	      case SDL_SCANCODE_C:
		life_seed += 100;
		return TRUE;

	      case SDL_SCANCODE_M:
		life_method ++;
		if (life_method > 31) life_method = 1;
		return TRUE;

	      case SDL_SCANCODE_J:
		life_jump = (life_jump + 1) & 31;
		return TRUE;

	      case SDL_SCANCODE_S:		/* Change Scale */
		life_scale = (life_scale & 3) + 1;
		return TRUE;

	    }
	    return TRUE;
	}
      }

      SDL_Delay (20);		/* Delay X millisec */
      delay -= 20;
    }
    return TRUE;
}


  /* Main program loop. */
void app_loop ()
{
    int cret;
	/* Main program loop */
    do {
      SDL_GetWindowSize (hWin, &vdux, &vduy);
      SDL_SetRenderDrawColor (sRender, 0, 0, 0, 255);
      SDL_RenderClear (sRender); 
		/* Ok redraw and render screen */
      app_Draw ();
      SDL_RenderPresent (sRender); 
		/* Check for keyboard/mouse/system events*/
      cret = app_event (1000);
      if (cret == FALSE) return;	/* Exit */
      sprintf (szTitle, "Seed:%d  #color:%d  RGB(%d,%d,%d)  Method:%d  Jump:%d  Scale:%d  CMD:SPACE=Next,ESC,rgbdntuxcjs.   %s",
	life_seed, life_maxcol, life_red, life_green, life_blue, life_method, life_jump, life_scale, szProg);
      SDL_SetWindowTitle (hWin, szTitle);
    } while (1);	/* until key/mouse hit */
}


void app_init ()
{
    int cret;
    cret = SDL_Init (SDL_INIT_VIDEO);
    if (cret != 0) {
      sdl_error ("SDL_Init");
    }

	/* Get desktop size */
    cret = SDL_GetCurrentDisplayMode (0, &dispMode);    
    if (cret != 0) {
      sdl_error ("SDL_GetCurrentDisplayMode");
    }
    vdux = dispMode.w - 10;
    vduy = dispMode.h - 60; 
    
    if (SDL_CreateWindowAndRenderer (vdux, vduy, SDL_WINDOW_RESIZABLE, &hWin, &sRender) < 0) {
      sdl_error ("SDL_CreateWindowAndRenderer");
    }
    if (sRender == NULL) {
      sdl_error ("sRender is NULL");
    }
    SDL_SetWindowTitle (hWin, szProg);
    SDL_GetWindowSize (hWin, &vdux, &vduy);
}


	/* Tidy up, quit. */
void app_end ()
{
    SDL_DestroyWindow (hWin);
    SDL_Quit ();
}

	/* Main program */
void main ()
{
    app_init ();
    app_loop ();
    app_end ();
}

