#include "chesseng1.h"
#include "moves.h"

#ifdef _DEBUG
u32 nbn;
#endif

/**************************************
 *
 */

u8 keepCol(u8* t,u8 idx,u8 cnt,u8 col)
{
	u8 i,j=idx;
	for (i=0;i<cnt;i++)
	{
		if (COL(t[idx+i])==col)
		{
			if (idx+i>j)
			{
				u8 tmp = t[j];
				t[j] = t[idx+i];
				t[idx+i] = tmp;
			}
			j+=1;
		}
	}
	return j-idx;
}

u8 keepRow(u8* t,u8 idx,u8 cnt,u8 row)
{
	u8 i,j=idx;
	for (i=0;i<cnt;i++)
	{
		if ((ROW(t[idx+i])==row))
		{
			if (idx+i>j)
			{
				u8 tmp = t[j];
				t[j] = t[idx+i];
				t[idx+i] = tmp;
			}
			j+=1;
		}
	}
	return j-idx;
}

u8 diag[8];

u8 keepDiag(u8* t,u8 idx,u8 cnt,u8 x,u8 c1,u8 r1,u8 c2,u8 r2)
{
	u8 c0,r0,d,i=0,p,j = idx;
	if (c1<c2)
	{
		c0 = c1;
		c1 = c2;
		r0 = r1;
		r1 = r2;
	}
	else
	{
		c0 = c2;
		r0 = r2;
	}
	d = r0<r1?1:255;
	for (;c0<=c1;c0++)
	{
		p = POS(r0,c0);
		if (p!=x)
			diag[i++] = p;
		r0+=d;
	}
	for (d=0;d<cnt;d++)
	{
		for (c0=0;(c0<i) && (diag[c0]!=t[idx+d]);c0++);
		if (c0<i)
		{
			u8 tmp = t[j];
			t[j++] = t[idx+d];
			t[idx+d] = tmp;
		}
	}
	return j-idx;
}

u8 pinned[32];

void computeControls(Board b)
{
	u8 *c,color,cc,x,y,n,i,colcheck,p,nn,idx = 0,pin,j;
	u16 offset,len,end;

	b->check = 0;
	b->chkp0 = b->chkp1 = b->chkm = DEAD;
	gm_memset(pinned,DEAD,sizeof(pinned));
	for (n=0;n<32;n++)
	{
		if ((p = b->pieces[n])==DEAD)
			continue;
		b->idx[n] = idx;
		nn = b->types[n];
		c = &b->ctrl[idx];
		cc = 0;
		color = COLOR(n);
		colcheck = (color==BLACK?1:2);
		pin = canPin(nn);

		for (i=0;i<8;i++)
		{
			if ((len = moves[nn][p][i][1])==0)
				break;
			offset = moves[nn][p][i][0];
			end = offset+len;
			x = DEAD;
			while (x==DEAD && offset<end)
			{
				y = c[cc++] = moves_dst[offset++];
				x = b->board[y];
			}
			if ((x!=DEAD) && (COLOR(x)!=color))
			{
				if (b->types[x]==T_KING)
				{
					b->check |= colcheck;
					if (pin)
					{
						b->chkp0 = n;
						y = DEAD;
						while (y==DEAD && offset<end)
							y = b->board[moves_dst[offset++]];
						if (y!=DEAD && COLOR(y)!=color)
							b->chkm = y;
					}
					else /* knight or pawn */
						b->chkp1 = n;
				}
				else if (pin)
				{
					y = DEAD;
					while (y==DEAD && offset<end)
						y = b->board[moves_dst[offset++]];
					if (y!=DEAD && b->types[y]==T_KING && COLOR(y)!=color) /* pinned ? */
						pinned[x] = n;
				}
			}
		}
		b->count[n] = cc;
		idx += cc;
	}
	b->max = idx;
	/* Restrict moves of pinned pieces */
	for (n=0;n<32;n++)
	{
		if (pinned[n]==DEAD)
			continue;
		p = b->pieces[KING+(n<BLACK?0:BLACK)];
		i = b->types[pinned[n]];
		x = b->pieces[n];
		y = b->pieces[pinned[n]];
		switch(b->types[n])
		{
			case T_WPAWN:
			case T_BPAWN:
				j = b->count[n];
				b->count[n] = 0;
				if (i==T_ROOK || (i==T_QUEEN && (COL(x)==COL(y)||ROW(x)==ROW(y))))
					break;
				else if (((b->types[n]==T_WPAWN)&&(y==x+7||y==x+9)) || ((b->types[n]==T_BPAWN)&&(y==x-7||y==x-9)))
				{
					if (j==2)
					{
						if (b->ctrl[b->idx[n]]!=y)
						{
							j = b->ctrl[b->idx[n]];
							b->ctrl[b->idx[n]] = y;
							b->ctrl[b->idx[n]+1] = j;
						}
						b->count[n] = 1;
					}
				}
				break;
			case T_KNIGHT:
				b->count[n] = 0;
				break;
			case T_ROOK:
				if (i==T_BISHOP)
					b->count[n] = 0;
				else if (i==T_QUEEN||i==T_ROOK)
				{
					if (COL(x)==COL(y))
						b->count[n] = keepCol(b->ctrl,b->idx[n],b->count[n],(u8)COL(x));
					else if (ROW(x)==ROW(y))
						b->count[n] = keepRow(b->ctrl,b->idx[n],b->count[n],(u8)ROW(x));
					else
						b->count[n] = 0;
				}
				break;
			case T_BISHOP:
				if (i==T_ROOK || (i==T_QUEEN && (COL(x)==COL(y)||ROW(x)==ROW(y))))
				{
					b->count[n] = 0;
					break;
				}
				b->count[n] = keepDiag(b->ctrl,b->idx[n],b->count[n],x,(u8)COL(p),(u8)ROW(p),(u8)COL(y),(u8)ROW(y));
				break;
			case T_QUEEN:
				if (i==T_QUEEN||i==T_ROOK)
				{
					if (COL(x)==COL(y))
					{
						b->count[n] = keepCol(b->ctrl,b->idx[n],b->count[n],(u8)COL(x));
						break;
					}
					else if (ROW(x)==ROW(y))
					{
						b->count[n] = keepRow(b->ctrl,b->idx[n],b->count[n],(u8)ROW(x));
						break;
					}
				}
				b->count[n] = keepDiag(b->ctrl,b->idx[n],b->count[n],x,(u8)COL(p),(u8)ROW(p),(u8)COL(y),(u8)ROW(y));
				break;
		}
	}
}

/**************************************
 *
 */

u16 curBoardAlloc;
u16 curBoardFree;

s16 getFreeBoard()
{
	u16 ret=curBoardAlloc++;
	if (curBoardAlloc==MAX_BACKBOARDS)
		curBoardAlloc=0;
	return free_bb[ret];
}

void freeBoard(s16 b)
{
	free_bb[curBoardFree++] = b;
	if (curBoardFree==MAX_BACKBOARDS)
		curBoardFree=0;
}

/**************************************
 *
 */

u16 curNodeAlloc;
u16 curNodeFree;

u16 getFreeNode()
{
	u16 ret=curNodeAlloc++;
	if (curNodeAlloc==MAX_NODES)
		curNodeAlloc=0;
	return free_nodes[ret];
}

void freeNode(u16 node)
{
	free_nodes[curNodeFree++] = node;
	if (ch_tree[node].bb>=0)
	{
		freeBoard(ch_tree[node].bb);
		ch_tree[node].bb = -1;
	}
	if (curNodeFree==MAX_NODES)
		curNodeFree=0;
}

/**************************************
 *
 */

void freeTree(u16 node,u8 all)
{
	u8 i,n=ch_tree[node].cnt;
	for (i=0;i<n;i++)
		freeTree(ch_tree[node].children[i],1);
	ch_tree[node].cnt=0;
	if (all)
		freeNode(node);
}

/**************************************
 *
 */

u8 maxdepth;
u8 lastsrc,lastdst;

void init(u8 lvl,u8 usebook)
{
	u16 i;
	const u8 itypes[] = {T_ROOK,T_KNIGHT,T_BISHOP,T_QUEEN,T_KING,T_BISHOP,T_KNIGHT,T_ROOK};

	curNodeAlloc = 0;
	curNodeFree = 0;
	for (i=0;i<MAX_NODES;i++)
	{
		ch_tree[i].index = i;
		ch_tree[i].bb = -1;
		ch_tree[i].cnt= 0;
		free_nodes[i] = i;
	}

	curBoardAlloc = 0;
	curBoardFree = 0;
	for (i=0;i<MAX_BACKBOARDS;i++)
		free_bb[i] = i;

	level = &levels[lvl];
	maxdepth=level->depth-1;
	nbmoves = 0;
	curnode = &ch_tree[getFreeNode()];
	curnode->cnt=0;
	curboard = &bboards[curnode->bb = getFreeBoard()];

	gm_memset(curboard->board,DEAD,64);
	for (i=0;i<8;i++)
	{
		curboard->pieces[i+BLACK] = POS(6,i);
		curboard->board[POS(6,i)] = i+BLACK;
		curboard->types[i+BLACK]  = T_BPAWN;
		curboard->pieces[i+WHITE] = POS(1,i);
		curboard->board[POS(1,i)] = i+WHITE;
		curboard->types[i+WHITE]  = T_WPAWN;
	}
	for (i=8;i<16;i++)
	{
		curboard->pieces[i+BLACK]   = POS(7,i-8);
		curboard->board[POS(7,i-8)] = i+BLACK;
		curboard->types[i+BLACK]    = itypes[i-8];
		curboard->pieces[i+WHITE]   = POS(0,i-8);
		curboard->board[POS(0,i-8)] = i+WHITE;
		curboard->types[i+WHITE]    = itypes[i-8];
	}
	curboard->castle=0xff; /* 2*4 bits set to 1 : Rooks + king + bit control */
	curboard->bonus = 0;
	computeControls(curboard);

	initBook(usebook);

	gm_memset(history,0,sizeof(history));
	histcnt = 0;
	moves50 = 0;
	reset   = 0;
	lastsrc = DEAD;
	phase	= OPENING;
}

/**************************************
 *
 */

u8 hasControl(Board b,u8 color,u8 pos)
{
    u8 i,*c = b->ctrl,j;
	/* find first not dead black piece */
	for (i=BLACK;i<32 && b->pieces[i]==DEAD;i++);
	/* for white, search from 0 to idx of this piece */
	if (color==WHITE)
	{
		j = b->idx[i];
		i = 0;
	}
	/* for black, search from idx of this piece to the end */
	else
	{
		i = b->idx[i];
		j = b->max;
	}
	for (;i<j && c[i]!=pos;i++);
	return i<j;
}

/**************************************
 *
 */

void completePawnMoves(Board b,u8 *out,u8* cc,u8 cp,u8 src)
{
	u8 i = COL(src), y = cp==WHITE?1:6, _cc = *cc, p;
	u8 pasy = cp==WHITE?4:3, passant = cp==WHITE ? b->passantb : b->passantw;
	u8 sy = src+(cp==WHITE?8:-8);
	u8 sy2 = src+(cp==WHITE?16:-16);
	u8 row = ROW(src);

	_cc = 0;
	if (b->board[sy]==DEAD)
	{
		out[_cc++]=sy;
		if (row==y&&b->board[sy2]==DEAD)
			out[_cc++]=sy2;
	}
	if (i>0)
	{
		p = b->board[sy-1];
		if ((p!=DEAD && COLOR(p)!=cp)||(row==pasy && passant==i-1))
			out[_cc++]=sy-1;
	}
	if (i<7)
	{
		p = b->board[sy+1];
		if ((p!=DEAD && COLOR(p)!=cp)||(row==pasy && passant==i+1))
			out[_cc++]=sy+1;
	}
	*cc = _cc;
}

/**************************************
 *
 */

void completeKingMoves(Board b,u8 *out,u8 *c,u8 *cc,u8 cp)
{
	u8 _cc=*cc;
	gm_memcpy(out,c,_cc);
	if ((cp==WHITE) && (b->castle&0x07) && (!isCheck(b,WHITE)))
	{
		if (((b->castle&0x05)==0x05)&&(b->pieces[Q_ROOK]!=DEAD)&&(b->board[1]==DEAD)&&(b->board[2]==DEAD)&&(b->board[3]==DEAD))
		{
			if (!(hasControl(b,BLACK,2) || hasControl(b,BLACK,3)))
				out[_cc++] = 2;
		}
		if (((b->castle&0x06)==0x06)&&(b->pieces[K_ROOK]!=DEAD)&&(b->board[5]==DEAD)&&(b->board[6]==DEAD))
		{
			if (!(hasControl(b,BLACK,5) || hasControl(b,BLACK,6)))
				out[_cc++] = 6;
		}
	}
	else if ((cp==BLACK) && (b->castle&0x70) && (!isCheck(b,BLACK)))
	{
		if (((b->castle&0x50)==0x50)&&(b->pieces[Q_ROOK+BLACK]!=DEAD)&&(b->board[57]==DEAD)&&(b->board[58]==DEAD)&&(b->board[59]==DEAD))
		{
			if (!(hasControl(b,WHITE,58) || hasControl(b,WHITE,59)))
				out[_cc++] = 58;
		}
		if (((b->castle&0x60)==0x60)&&(b->pieces[K_ROOK+BLACK]!=DEAD)&&(b->board[61]==DEAD)&&(b->board[62]==DEAD))
		{
			if (!(hasControl(b,WHITE,61) || hasControl(b,WHITE,62)))
				out[_cc++] = 62;
		}
	}
	*cc=_cc;
}

/***************************************************************
 * verify
 *
 * return 0 if move from src to dst is possible (not check test)
 */

u8 verify(Board b,u8 src,u8 dst)
{
	u8 p = b->board[src], cp = COLOR(p), out[10];
	u8 *c = &b->ctrl[b->idx[p]], cc = b->count[p],i,tp = b->types[p];

	if (isPawn(tp))
	{
		completePawnMoves(b,out,&cc,cp,src);
		c = out;
	}
	else if (tp==T_KING)
	{
		completeKingMoves(b,out,c,&cc,cp);
		c = out;
	}

	for (i=0;i<cc&&c[i]!=dst;i++);
	if (i<cc)
	{
		u8 pp = b->board[dst];
		return ((pp==DEAD)||(COLOR(pp)!=cp))?0:1;
	}
	return 1;
}

/**************************************
 *
 */

u16 children[MAX_ROOT];
u8 sorted[MAX_ROOT];
u16 root_children[MAX_ROOT];

u8 possibleMoves(CH_NODE *node,Board b,u8 color,u8 maxkeep,u8 newroot)
{
	u8 i,j,k,n,p,cc,*c,pp,out[10],mink=0,tp,cnt,cnt0;
	s16 value,vmink = 0;
	s16 work;
	u16 *cptr;
	u8 mxk = maxkeep;

	if (node==curnode)
	{
		if (newroot==0)
			return MIN(node->cnt,maxkeep);
		if (newroot>1)
			return newroot;
		maxkeep = MAX_ROOT;
	}
	if (node!=NULL && node->cnt>0)
	{
		CH_NODE * cn;
		cnt0 = node->cnt;
		for (i=0;i<cnt0;i++)
		{
			cn = &ch_tree[node->children[i]];
			children[i] = cn->index;
			if (cn->bb<0)
			{
				cn->bb = getFreeBoard();
				copy(&bboards[cn->bb],b);
				play(cn->src,cn->dst,&bboards[cn->bb],T_QUEEN,0);
			}
		}
		if (cnt0==maxkeep)
			return cnt0;
		cnt = cnt0;
	}
	else
		cnt0 = cnt = 0;

	work = getFreeBoard();
	for (i=0;i<16;i++)
	{
		/* Verify User action (reset, flip...) */
		userAction();
		if (reset)
			return 0;

		n = i+color;
		p = b->pieces[n];
		if (p==DEAD)
			continue;

 		cc = b->count[n];
		tp = b->types[n];
		c  = &b->ctrl[b->idx[n]];
		if (isPawn(tp)) /* move with no control */
		{
			completePawnMoves(b,out,&cc,color,p);
			c = out;
		}
		else if (tp==T_KING) /* castle moves */
		{
			completeKingMoves(b,out,c,&cc,color);
			c = out;
		}

		for (j=0;j<cc;j++)
		{
			/* Verify User action (reset, flip...) */
			userAction();
			if (reset)
				return 0;

			pp = b->board[c[j]];
			if (pp!=DEAD&&(COLOR(pp)==color))
				continue;

			for (k=0;k<cnt0;k++)
			{
				CH_NODE *cn = &ch_tree[children[k]];
				if (cn->src==p && cn->dst==c[j])
					break;
			}
			if (k<cnt0)
				continue;

			copy(&bboards[work],b);
			n = play(p,c[j],&bboards[work],T_QUEEN,0);

			if (isCheck(&bboards[work],color))
				continue;

			if (maxkeep==0) /* check mat test from eval function */
			{
				freeBoard(work);
				return 1;
			}

			n = n || isCheck(&bboards[work],INV(color));
			value = eval(&bboards[work],color);

			if (cnt<maxkeep)
			{
				u16 idx = getFreeNode();
				CH_NODE *cn = &ch_tree[idx];
				children[cnt++] = idx;
				cn->src   = p;
				cn->dst   = c[j];
				cn->quiet = n;
				cn->bb    = work;
				cn->value = value;
				cn->cnt	  = 0;
				work      = getFreeBoard();
			}
			else if (value>vmink)
			{
				CH_NODE *cn = &ch_tree[children[mink]];
				s16 tmp	  = cn->bb;
				cn->src   = p;
				cn->dst   = c[j];
				cn->quiet = n;
				cn->bb    = work;
				cn->value = value;
				cn->cnt	  = 0;
				work      = tmp;
			}
			else /* no modification */
				continue;
			if (cnt==maxkeep)
			{
				mink = 0;
				vmink = ch_tree[children[0]].value;
				for (k=1;k<maxkeep;k++)
				{
					if (vmink>ch_tree[children[k]].value)
					{
						vmink=ch_tree[children[k]].value;
						mink = k;
					}
				}
			}
		}
	}
	freeBoard(work);

	if (maxkeep==0) /* check mat test */
		return 0;

	node->cnt = cnt;
	cptr = node==curnode?root_children:node->children;
	if (cnt==1)
	{
		cptr[0] = children[0];
		if (cnt<maxkeep)
			ch_tree[children[0]].quiet=1;
	}
	/* sort first level for time mngt */
	else if (node==curnode)
	{
		gm_memset(sorted,1,cnt);
		i = cnt;
		while (i>0)
		{
			vmink=MAX+1;
			for (k=0;k<cnt;k++)
			{
				if (sorted[k] && vmink>ch_tree[children[k]].value)
				{
					vmink=ch_tree[children[k]].value;
					mink = k;
				}
			}
			cptr[--i] = children[mink];
			sorted[mink] = 0;
		}
#ifdef _DEBUG
		for (i=0;i<cnt;i++)
		{
			char buf[32];
			sprintf(buf,"%c%d%c%d %d",COL(ch_tree[cptr[i]].src)+'a',ROW(ch_tree[cptr[i]].src)+1,COL(ch_tree[cptr[i]].dst)+'a',ROW(ch_tree[cptr[i]].dst)+1,ch_tree[cptr[i]].value);
			LOG(buf)
		}
#endif
	}
	else
		gm_memcpy(cptr,children,sizeof(u16)*cnt);
	return MIN(mxk,cnt);
}

/**************************************
 *
 */

u8 cursrc,curdst;
u8 drawsrc,drawdst;
s16 max_clock;

void searchMove(u8 color)
{
	s16 ch_clock = -1;
	s16	value,next;
#ifdef _DEBUG
	long t0,t1;
#else
	u32 t0,t1;
#endif
	u8	i,j,timeok=1;

	if (nbmoves<2)
		max_clock = getChessClock();

	if (useBook())
	{
		getBookEntry();
		return;
	}

	ch_clock = getChessClock();
	t0 = getMachineTime();

#ifdef _DEBUG
	nbn=0;
	LOG("TIME")
	LOGI(max_clock)
	LOGI(ch_clock)
#endif

	if (ch_clock>0) /* -1 = infinite time */
	{
		switch(phase)
		{
			case OPENING:
				ch_clock = (max_clock*2)/(5*QUEENMV);
				break;
			case MIDDLEGAME:
				if (nbmoves<48+QUEENMV)
				{
					ch_clock = (s16)(((u32)max_clock*7)/(10*34));
					break;
				}
			case ENDGAME:
				ch_clock = ch_clock/10;
				break;
		}
		ch_clock+=getTimeBonus();
	}

#ifdef _DEBUG
LOGI(ch_clock)
#endif

	cursrc=DEAD;
	drawsrc=DEAD;
	drawdst=DEAD;
	if (histcnt>15) /* 3 fold repetitions */
	{
		u8 a = history[histcnt-16];
		u8 b = history[histcnt-15];
		u8 c = history[histcnt-14];
		u8 d = history[histcnt-13];
		if (a == history[histcnt-11] &&
			a == history[histcnt-8]  &&
			a == history[histcnt-3]  &&
			b == history[histcnt-12] &&
			b == history[histcnt-7]  &&
			b == history[histcnt-4]  &&
			c == history[histcnt-9]  &&
			c == history[histcnt-6]  &&
			c == history[histcnt-1]  &&
			d == history[histcnt-10] &&
			d == history[histcnt-5]  &&
			d == history[histcnt-2])
		{
			drawsrc = a;
			drawdst = b;
		}
	}

	value = curnode->value;

	if (ch_clock<0) /* -1 = infinite time */
	{
		next = compute(color,maxdepth,level->initdepth,maxdepth,0,curnode,-MAX,MAX,1);
		if (value-next>HELP_WANTED)
		{
			for (i=lvl_keep[maxdepth]+1;i<=curnode->cnt;i++)
			{
				compute(color,maxdepth,level->initdepth,3,i,curnode,-MAX,MAX,i);
				freeTree(root_children[i-1],0);
			}
		}

	}
	else
	{
		u8 lvlk = MIN(lvl_keep[maxdepth],curnode->cnt);
		next = compute(color,maxdepth,0,3,0,curnode,-MAX,MAX,1);
		t1 = getMachineTime();
		if ((curnode->cnt<2) || (next==MAX) || ((u32)(t1-t0)>=(u32)ch_clock))
			timeok = 0;
		j = lvlk;
		for (i=5;i<level->depth && timeok;i+=2)
		{
			for (j=1;j<=lvlk && timeok;j++)
			{
				next = compute(color,maxdepth,level->initdepth,i,j,curnode,-MAX,MAX,0);
				if (reset||next==MAX)
					timeok = 0;
				t1 = getMachineTime();
				if ((u32)(t1-t0)>=(u32)ch_clock)
				{
#ifdef _DEBUG
					char buf[32];
					sprintf(buf,"BREAK ON TIME %d %d",i,j);
					LOG(buf);
#endif
					timeok=0;
				}
			}
		}
		/* extended time if bad result found */
		ch_clock+=(ch_clock>>1);
		timeok = value-next>HELP_WANTED;
		for (i=lvl_keep[maxdepth]+1;i<=curnode->cnt && timeok;i++)
		{
#ifdef _DEBUG
			LOG("HELP")
			LOGI(value)
			LOGI(next)
#endif
			next = compute(color,maxdepth,level->initdepth,3,i,curnode,-MAX,MAX,i);
			freeTree(root_children[i-1],0);

			if (reset)
				break;
			t1 = getMachineTime();
			if ((next!=-MAX) && ((u32)(t1-t0)>=(u32)ch_clock))
			{
#ifdef _DEBUG
				char buf[32];
				sprintf(buf,"BREAK ON EXTENDED TIME %d",i);
				LOG(buf);
#endif
				timeok = 0;
			}
		}
	}

#ifdef _DEBUG
	{
		char buf[32];
		time(&t1);
		sprintf(buf,"time %d %d",t1-t0,nbn);
		LOG(buf);
	}
#endif
}

/**************************************
 *
 */

#ifdef _DEBUG
int xxx=0;
#endif

s16 compute(u8 color,u8 depth,u8 mindepth,u8 sdepth,u8 skeep,CH_NODE*node,s16 alpha,s16 beta,u8 newroot)
{
	u8  j,n,nbk = 0;
	s16 min = -MAX-1;
	u16 *cptr = node==curnode?root_children:node->children;

	nbk = possibleMoves(node,&bboards[node->bb],color,lvl_keep[depth],newroot);

	if (nbk==0||reset)
	{
		if (reset)
			return 0;
		nbk = getPatMat(&bboards[node->bb],color);
		if (nbk==MAT||nbk==NOPM)
			return -MAX;
		return 0;
	}

	if (nbk==1&&depth==maxdepth)
	{
		CH_NODE *cn = &ch_tree[cptr[0]];
		cursrc = cn->src;
		curdst = cn->dst;
		return cn->value;
	}
	else
	{
		for (j=0;j<nbk;j++)
		{
			CH_NODE *cn = &ch_tree[cptr[j]];
			/* Verify User action (reset, flip...) */
			userAction();
			if (reset)
				return 0;

			n = (depth==mindepth) && ((mindepth%2)==0) && (mindepth>0) && (cn->quiet) && (sdepth>0);
#ifdef _DEBUG
			line[nbl++] = cn->src;
			line[nbl++] = cn->dst;
#endif
			if (n)
				mindepth -=2;
			if ((depth==maxdepth) && (cn->src==drawsrc) && (cn->dst==drawdst)) /* 3 fold repetitions */
				cn->value = 0;
			else if (depth==mindepth||sdepth==0||(skeep>0 && j+1!=skeep))
				cn->value = cn->value;
			else
				cn->value = -compute(INV(color),(u8)(depth-1),mindepth,(u8)(sdepth-1),0,cn,(s16)(-beta),(s16)(-alpha),0);
			if (depth<maxdepth-3)
			{
				freeBoard(cn->bb);
				cn->bb = -1;
			}
			if (n)
				mindepth+=2;
#ifdef _DEBUG
			logLine(0,cn->value);
			nbl-=2;
			if (cn->bb>0)
			{
				xxx=1;
				eval(&bboards[cn->bb],color);
				xxx=0;
			}
#endif
			if (cn->value>min)
			{
				if (depth==maxdepth)
				{
					cursrc=cn->src;
					curdst=cn->dst;
				}
				min = cn->value;
			}
		}
	}

	if (min>alpha)
		alpha = min;
	if (alpha>=beta)
		return alpha;
	return min;
}

/**************************************
 *
 */

u8 around(Board b,u8 pos,u8 color)
{
	u8 nb = 0, c = COL(pos), r = ROW(pos), i, ret = 0;
	u8 tab[8], flg = 0;

	if (color==BLACK && r==7)
	{
		if (c>0)
			tab[nb++] = b->board[pos-9];
		if (c<7)
			tab[nb++] = b->board[pos-7];
		tab[nb++] = b->board[pos-8];
		ret = 2;
	}
	else if (color==WHITE && r==0)
	{
		if (c>0)
			tab[nb++] = b->board[pos+7];
		if (c<7)
			tab[nb++] = b->board[pos+9];
		tab[nb++] = b->board[pos+8];
		ret = 2;
	}
	else
	{
		if (c>0)
		{
			tab[nb++] = b->board[pos-1];
			if (r>0)
				tab[nb++] = b->board[pos-9];
			if (r<7)
				tab[nb++] = b->board[pos+7];
		}
		if (c<7)
		{
			tab[nb++] = b->board[pos+1];
			if (r>0)
				tab[nb++] = b->board[pos-7];
			if (r<7)
				tab[nb++] = b->board[pos+9];
		}
		if (r>0)
			tab[nb++] = b->board[pos-8];
		if (r<7)
			tab[nb++] = b->board[pos+8];
		flg = 1;
	}

	for (i=0;i<nb;i++)
		if (tab[i]!=DEAD && COLOR(tab[i])==color && isPawn(b->types[tab[i]]))
			ret+=2;

	if (flg)
		ret>>=1;

	return ret;
}

/**************************************
 *
 */

u8 play(u8 src,u8 dst,Board b,u8 promo,u8 dobook)
{
	u8 pp = b->board[dst],p,castle=0,i;
	u8 colsrc = COL(src),rowsrc = ROW(src);

	if (pp!=DEAD)
		b->pieces[pp] = DEAD;

	b->bonus>>=1;
	p = b->board[src];
	b->board[src] = DEAD;
	b->board[dst] = p;
	b->pieces[p] = dst;
	if (b->types[p]==T_WPAWN)
	{
		b->passantw = dst-src==16 ? p : NOPASSANT;
		if (COL(dst)!=colsrc && pp==DEAD) /* en passant */
		{
			i = POS(4,COL(dst));
			b->pieces[b->board[i]] = DEAD;
			b->board[i] = DEAD;
		}
		b->passantb = NOPASSANT;
		if (ROW(dst)==7) /* Promotion */
			b->types[p] = promo;
		/* safe castle */
		if ((b->castle&0x05)==0x05 && colsrc<3 && rowsrc==1)
			b->bonus-=phases[phase][CASTLE_QPAWN];
		else if ((b->castle&0x06)==0x06 && colsrc>4 && rowsrc==1)
			b->bonus-=phases[phase][CASTLE_KPAWN];
	}
	else if (b->types[p]==T_BPAWN)
	{
		b->passantb = src-dst==16 ? p-BLACK : NOPASSANT;
		if (COL(dst)!=COL(src) && pp==DEAD) /* en passant */
		{
			i = POS(3,COL(dst));
			b->pieces[b->board[i]] = DEAD;
			b->board[i] = DEAD;
		}
		b->passantw = NOPASSANT;
		if (ROW(dst)==0) /* Promotion */
			b->types[p] = promo;
		/* safe castle */
		if ((b->castle&0x50)==0x50 && colsrc<3 && rowsrc==6)
			b->bonus+=phases[phase][CASTLE_QPAWN];
		else if ((b->castle&0x60)==0x60 && colsrc>4 && rowsrc==6)
			b->bonus+=phases[phase][CASTLE_KPAWN];
	}
	/* Castle */
	if (p==KING)
	{
		if (dst-src==2)
		{
			b->board[7] = DEAD;
			b->board[5] = K_ROOK;
			b->pieces[K_ROOK] = 5;
			castle = 1;
		}
		else if (src-dst==2)
		{
			b->board[0] = DEAD;
			b->board[3] = Q_ROOK;
			b->pieces[Q_ROOK] = 3;
			castle = 1;
		}
	}
	else if (p==KING+BLACK)
	{
		if (dst-src==2)
		{
			b->board[63] = DEAD;
			b->board[61] = K_ROOK+BLACK;
			b->pieces[K_ROOK+BLACK] = 61;
			castle = 1;
		}
		else if (src-dst==2)
		{
			b->board[56] = DEAD;
			b->board[59] = Q_ROOK+BLACK;
			b->pieces[Q_ROOK+BLACK] = 59;
			castle = 1;
		}
	}
	computeControls(b);

	switch(p)
	{
		case Q_ROOK:			/* BBBB WWWW */
			if (b->castle&0x05)
			{
				b->bonus  -= phases[phase][CASTLE_QROOK];
				b->castle &= 0xfe;	/* 1111 1110 */
			}
			break;
		case Q_ROOK+BLACK:
			if (b->castle&0x50)
			{
				b->bonus  += phases[phase][CASTLE_QROOK];
				b->castle &= 0xef;	/* 1110 1111 */
			}
			break;
		case K_ROOK:
			if (b->castle&0x06)
			{
				b->bonus  -= phases[phase][CASTLE_KROOK];
				b->castle &= 0xfd;	/* 1111 1101 */
			}
			break;
		case K_ROOK+BLACK:
			if (b->castle&0x60)
			{
				b->bonus  += phases[phase][CASTLE_KROOK];
				b->castle &= 0xdf;	/* 1101 1111 */
			}
			break;
		case KING:
			if (castle)
			{
				b->castle &= 0xf8;
				b->bonus  += phases[phase][CASTLE];
			}
			else if (b->castle&0x04)
			{
				b->castle &= 0xf0;	/* 1111 0000 */
				b->bonus  -= phases[phase][CASTLE_KING];
			}
			break;
		case KING+BLACK:
			if (castle)
			{
				b->castle &= 0x8f;
				b->bonus  -= phases[phase][CASTLE];
			}
			else if (b->castle&0x40)
			{
				b->castle &= 0x0f;	/* 0000 1111 */
				b->bonus  += phases[phase][CASTLE_KING];
			}
			break;
	}
	if (useBook() && dobook)
		playBook(src,dst);
	return pp!=DEAD;
}

/**************************************
 *
 */

void playMove(u8 src,u8 dst,u8 promo,u8 dobook,u8 newroot)
{
	u8 pp = curboard->board[dst],i;
	CH_NODE *cn = NULL;
	u16 *cptr = newroot?root_children:curnode->children;

	if (pp!=DEAD)
		moves50 = 0;
	else
		moves50++;

	lastsrc = src;
	lastdst = dst;

	for (i=0;i<curnode->cnt;i++)
	{
		u16 j = cptr[i];
		CH_NODE *nd = &ch_tree[j];
		if (nd->src!=src || nd->dst!=dst)
			freeTree(j,1);
		else
			cn = nd;
	}
	curnode->cnt = 0;
	if (cn!=NULL)
	{
		freeNode(curnode->index);
		curboard = &bboards[cn->bb];
		curnode = cn;
		if (useBook() && dobook)
			playBook(cn->src,cn->dst);
	}
	else
	{
		play(src,dst,curboard,promo,dobook);
		curnode->value = eval(curboard,(u8)COLOR(curboard->pieces[dst]));
	}

	history[histcnt++] = src;
	history[histcnt++] = dst;
	if (phase==OPENING)
	{
		if (!useBook() && nbmoves>=QUEENMVLIM)
			phase = MIDDLEGAME;
	}
	else if (phase==MIDDLEGAME)
	{
		if (curboard->pieces[QUEEN]==DEAD || curboard->pieces[QUEEN+BLACK]==DEAD)
		{
			u8 nw=0,nb=0;
			for(i=8;i<16;i++)
			{
				if (curboard->pieces[i]!=DEAD)
					nw++;
				if (curboard->pieces[i+BLACK]!=DEAD)
					nb++;
			}
			if (nw<4 || nb<4)
				phase = ENDGAME;
		}
	}
#ifdef _DEBUG
LOG("PHASE");
LOGI(phase);
#endif
}

/**************************************
 *
 */

u8 getPatMat(Board b,u8 color)
{
	if (possibleMoves(NULL,b,INV(color),0,0))
	{
		if (b==curboard && moves50>100)
			return MOVES50;
		if (b==curboard && histcnt>18)
		{
			u8 a = history[histcnt-18];
			u8 b = history[histcnt-17];
			u8 c = history[histcnt-16];
			u8 d = history[histcnt-15];
			if (a == history[histcnt-13] &&
				a == history[histcnt-10] &&
				a == history[histcnt-5]  &&
				a == history[histcnt-2]  &&
				b == history[histcnt-14] &&
				b == history[histcnt-9]  &&
				b == history[histcnt-6]  &&
				b == history[histcnt-1]  &&
				c == history[histcnt-11] &&
				c == history[histcnt-8]  &&
				c == history[histcnt-3]  &&
				d == history[histcnt-12] &&
				d == history[histcnt-7]  &&
				d == history[histcnt-4]
				)
			return THREEFOLD;
		}
		return NOPM;
	}
	if (((color==WHITE)&&isCheck(b,BLACK))||((color==BLACK)&&isCheck(b,WHITE)))
			return MAT;
	return PAT;
}

/**************************************
 *
 */

u8 nbattack[32];
u8 nbdefense[32];
u8 defense[32];
s16 minattack[32];
s16 multiattack[32];
s16 maxattack[32];
u8 nbmulti[32];
u8 blackattack[64];
u8 whiteattack[64];
u8 overload[32];
u8 locks[32];
u8 pstructw[8];
u8 pstructb[8];

#define GAIN(a) { if ((a)>v_pieces2) { v_minus+=v_pieces2; v_pieces2=a; } else v_minus+=a; }

s16 eval(Board b,u8 color)
{
	u8 i,j,*c,cc,p,cj,k,coli;
	s16 ci;
	s16 v,vi,vp;
	u8 bcheck = isCheck(b,BLACK);
	u8 wcheck = isCheck(b,WHITE);
	s16 v_pieces   = 0;
	s16 v_pieces2  = 0;
	s16 v_menace   = 0;
	s16 v_plus     = 0;
	s16 v_minus    = 0;
	s16 v_cases    = 0;
	s16 v_locked   = 0;
	s16 v_bonus    = (bcheck?1:(wcheck?-1:0))*phases[phase][CHECK]+b->bonus;
	s16 v_overload = 0;
	s16 v_unprotect= 0;
	s16 v_struct   = 0;
	s16 v_king     = 0;
	u8 *attack;
#ifdef _DEBUG
	nbn++;
#endif

	/* Check mat ? */
	i = getPatMat(b,color);
	if (i==MAT)
		return MAX;
	else if (i==PAT)
		return 0;

	gm_memset(whiteattack,0,sizeof(whiteattack));
	gm_memset(blackattack,0,sizeof(blackattack));
	gm_memset(nbattack,0,sizeof(nbattack));
	gm_memset(nbdefense,0,sizeof(nbdefense));
	gm_memset(nbmulti,0,sizeof(nbmulti));
	gm_memset(maxattack,0,sizeof(maxattack));

	for (i=0;i<32;i++)
	{
		if (b->pieces[i] == DEAD)
			continue;
		cc = b->count[i];
		c = &b->ctrl[b->idx[i]];
		attack = i<BLACK?whiteattack:blackattack;
		for (j=0;j<cc;j++)
			attack[c[j]] = 1;
	}

	/* Control */
	for (i=0;i<32;i++)
	{
		if (b->pieces[i] == DEAD)
			continue;
		coli = COLOR(i);
		vi = vpieces[b->types[i]];
		ci = i>=BLACK ? -1 : 1;
		v_pieces += ci*vi;
		cc = b->count[i];
		c = &b->ctrl[b->idx[i]];
		k=(cc==0?0:isPiece(b->types[i]));
		for (j=0;j<cc;j++)
		{
			cj = c[j];
			v_cases += ci*vboard[cj];
			p = b->board[cj];
			if (p!=DEAD)
			{
				if (COLOR(p)!=coli)
				{
					if (nbattack[p]==0 || vi<minattack[p])
						minattack[p] = vi;
					nbattack[p]+=1;
					if (b->types[p]!=T_KING)
					{
						vp = vpieces[b->types[p]];
						if (vp>vi)
						{
							if (nbmulti[i]==0 || vp<multiattack[i])
								multiattack[i] = vp;
							nbmulti[i]+=1;
						}
						if (vp>maxattack[i])
							maxattack[i] = vp;
					}
				}
				else
				{
					nbdefense[p]+=1;
					defense[p] = i;
					if (b->types[p]==T_ROOK && b->types[i]==T_ROOK)
						v_bonus+=ci*phases[phase][ROOKLINK];
				}
			}
			if (k && (p==DEAD || COLOR(p)==coli))
			{
				if (((i<BLACK && !blackattack[cj])||(i>=BLACK && !whiteattack[cj])))
					k=0;
			}
		}
		if (k)
		{
			v_locked -= ci*locked[b->types[i]];
			locks[i] = k;
		}
	}

	/* Pieces attacked */
	if (b->chkp0!=DEAD && b->chkp1!=DEAD) /* Dbl check */
	{
		if (maxattack[b->chkp0]||maxattack[b->chkp1])
		{
			vi = maxattack[b->chkp0]-vpieces[b->types[b->chkp0]];
			vp = maxattack[b->chkp1]-vpieces[b->types[b->chkp1]];
			v_plus = vi>vp ? vi:vp;
		}
	}
	else if (bcheck||wcheck)
	{
		p = b->chkp0!=DEAD ? b->chkp0 : b->chkp1;
#ifdef _DEBUG
if (xxx)
{
LOG("chkp")
LOGI(p);
LOGI(nbattack[p]);
LOGI(nbdefense[p]);
}
#endif
		vp = vpieces[b->types[p]];
		if (nbattack[p])
		{
			if (!nbdefense[p])
				v_pieces2 = vp;
			else if (minattack[p]<vp)
				v_pieces2 = vp-minattack[p];
		}
		else if (maxattack[p])
			v_plus = maxattack[p]-vp;
	}
	else
	{
		for (i=0;i<BLACK;i++)
		{
			k = i+color;
			j = b->types[k];
			if ((b->pieces[k]==DEAD)||(j==T_KING)||!nbattack[k])
				continue;
			vi = vpieces[j];
			if (!nbdefense[k])
				GAIN(vi)
			else if (minattack[k]<vi)
				GAIN(vi-minattack[k])
		}
		/* locked and attacked pieces */
		for (i=8;i<BLACK;i++)
		{
			k = i+(BLACK-color);
			j = b->types[k];
			if ((b->pieces[k]==DEAD)||(j==T_KING)||!nbattack[k]||!locks[k])
				continue;
			vi = vpieces[j];
			if (!nbdefense[k]||minattack[k]<vi)
				v_plus += vi-minattack[k];
		}
	}

	/* En passant */
	if (b->passantw)
	{
		k=b->passantw;
		j=0;
		while (k!=1)
		{
			k>>=1;
			j++;
		}
		k = COL(b->pieces[j]);
		for (i=0;i<8;i++)
		{
			j = b->pieces[i+BLACK];
			if (j==DEAD||ROW(j)!=2)
				continue;
			if (((k>0)&&(COL(j)==k-1))||((k<7)&&(COL(j)==k+1)))
			{
				if (!whiteattack[j])
				{
					GAIN(vpieces[T_WPAWN])
					break;
				}
			}
		}
	}
	else if (b->passantb)
	{
		k=b->passantb;
		j=0;
		while (k!=1)
		{
			k>>=1;
			j++;
		}
		k = COL(b->pieces[j+BLACK]);
		for (i=0;i<8;i++)
		{
			j = b->pieces[i];
			if (j==DEAD||ROW(j)!=5)
				continue;
			if (((k>0)&&(COL(j)==k-1))||((k<7)&&(COL(j)==k+1)))
			{
				if (!blackattack[j])
				{
					GAIN(vpieces[T_BPAWN])
					break;
				}
			}
		}
	}

	/* Pieces hidden by king while check */
	if (b->chkm!=DEAD)
	{
		vp = vpieces[b->types[b->chkp0]];
		vi = vpieces[b->types[b->chkm]];
		if (!nbdefense[b->chkm])
			v_plus += vi;
		else if (vp<vi)
			v_plus += vi-vp;
	}

	/* Multiple attacks on a bigger piece */
	for (i=0;i<BLACK;i++)
	{
		vp = vpieces[b->types[k = i+color]];
		if (nbmulti[k]>1 && (!nbattack[k] || (nbdefense[k] && minattack[k]>vp)))
			v_plus += (multiattack[k]-vp);
	}

	if (color==WHITE)
	{
		v_menace = v_plus-v_minus;
		v_pieces2 = -v_pieces2;
	}
	else
		v_menace = v_minus-v_plus;

	/* Unprotected pieces */
	for (i=0;i<BLACK;i++)
	{
		if ((b->pieces[i]==DEAD)||(b->types[i]==T_KING))
			continue;
		if (!nbdefense[i])
			v_unprotect -= (nbattack[i]?2:1)*unprotected[b->types[i]];
	}

	for (i=BLACK;i<32;i++)
	{
		if ((b->pieces[i]==DEAD)||(b->types[i]==T_KING))
			continue;
		if (!nbdefense[i])
			v_unprotect += (nbattack[i]?2:1)*unprotected[b->types[i]];
	}

	/* overload */
	gm_memset(overload,0,sizeof(overload));

	for (i=0;i<32;i++)
	{
		if (b->types[i]!=T_KING&&nbdefense[i]==1)
			overload[defense[i]]++;
	}
	for (i=0;i<BLACK;i++)
	{
		if (overload[i]>1)
			v_overload-=overload[i];
	}
	for (i=BLACK;i<32;i++)
	{
		if (overload[i]>1)
			v_overload+=overload[i];
	}

	/* Queen mallus if move too early */
	i = b->pieces[QUEEN];
	if ((i!=DEAD)&&(i!=POS(0,3))&&(nbmoves<QUEENMVLIM))
		v_bonus-=phases[phase][QUEENMV];
	i = b->pieces[QUEEN+BLACK];
	if ((i!=DEAD)&&(i!=POS(7,3))&&(nbmoves<QUEENMVLIM))
		v_bonus+=phases[phase][QUEENMV];

	/* Pawns struct */
	gm_memset(pstructw,0,sizeof(pstructw));
	gm_memset(pstructb,0,sizeof(pstructb));

	for (i=0;i<8;i++)
	{
		p = b->pieces[i];
		if (p==DEAD || b->types[i]!=T_WPAWN)
			continue;
		pstructw[COL(p)]++;
		v_struct+=pawn_bonus[phase][ROW(p)];
	}
	for (i=0;i<8;i++)
	{
		p = b->pieces[i+BLACK];
		if (p==DEAD || b->types[i+BLACK]!=T_BPAWN)
			continue;
		pstructb[COL(p)]++;
		v_struct-=pawn_bonus[phase][7-ROW(p)];
	}

	for (i=0;i<8;i++)
	{
		if (pstructw[i]>1)
			v_struct-=phases[phase][DBL_PAWN];
		if (((i>0?pstructw[i-1]:0)+(i<7?pstructw[i+1]:0))==0)
			v_struct-=pstructw[i]*phases[phase][ALONE_PAWN];
		if (pstructb[i]==0)
			v_struct+=phases[phase][PASSED_PAWN];
	}
	for (i=0;i<8;i++)
	{
		if (pstructb[i]>1)
			v_struct+=phases[phase][DBL_PAWN];
		if (((i>0?pstructb[i-1]:0)+(i<7?pstructb[i+1]:0))==0)
			v_struct+=pstructb[i]*phases[phase][ALONE_PAWN];
		if (pstructw[i]==0)
			v_struct-=phases[phase][PASSED_PAWN];
	}

	/* Kings protection */
	v_king += around(b,b->pieces[KING],WHITE);
	v_king -= around(b,b->pieces[KING+BLACK],BLACK);

	/* overall evaluation */
	v =		v_pieces	* phases[phase][PIECE]
		+	v_pieces2	* phases[phase][PIECE]
		+	v_menace	* phases[phase][MENACE]
		+	v_cases		* phases[phase][CASE]
		+	v_locked	* phases[phase][LOCKED]
		+	v_bonus		* phases[phase][BONUS]
		+	v_struct	* phases[phase][PAWNSTRUCT]
		+	v_overload	* phases[phase][OVERLOAD]
		+	v_unprotect	* phases[phase][UNPROTECT]
		+	v_king		* phases[phase][KINGPROTECT];

#ifdef _DEBUG
	if (xxx || b==curboard)
	{
		char buf[256];
		sprintf(buf,"V=%d p=%d(%d) p2=%d(%d) m=%d(%d) c=%d(%d) l=%d(%d) b=%d(%d) s=%d(%d) o=%d(%d) u=%d(%d) k=%d(%d)",
				v,
				v_pieces,    v_pieces    * phases[phase][PIECE],
				v_pieces2,   v_pieces2   * phases[phase][PIECE],
				v_menace,    v_menace    * phases[phase][MENACE],
				v_cases,     v_cases     * phases[phase][CASE],
				v_locked,    v_locked    * phases[phase][LOCKED],
				v_bonus,     v_bonus     * phases[phase][BONUS],
				v_struct,    v_struct    * phases[phase][PAWNSTRUCT],
				v_overload,  v_overload  * phases[phase][OVERLOAD],
				v_unprotect, v_unprotect * phases[phase][UNPROTECT],
				v_king,      v_king      * phases[phase][KINGPROTECT]);
		LOG(buf);
	}
#endif

	return color==WHITE?v:-v;
}
