#!/usr/bin/env bash
# dem_upload.sh
# Usage: ./dem_upload.sh filelist.txt output.csv
# filelist.txt: one path per line; blank lines and lines starting with # are ignored.

set -uo pipefail

INPUT_LIST="${1:-}"
OUTPUT_CSV="${2:-urls.csv}"
API_URL="https://tmpfiles.org/api/v1/upload"

if [[ -z "$INPUT_LIST" || ! -f "$INPUT_LIST" ]]; then
  echo "Usage: $0 <filelist.txt> [output.csv]" >&2
  exit 1
fi

# CSV header
echo "local_path,url,status,message" > "$OUTPUT_CSV"

have_jq=0
if command -v jq >/dev/null 2>&1; then
  have_jq=1
fi

# Read list line-by-line (preserves spaces)
while IFS= read -r path || [[ -n "$path" ]]; do
  # skip blanks/comments
  [[ -z "$path" || "$path" =~ ^[[:space:]]*# ]] && continue

  # Trim surrounding quotes/spaces
  path="${path%\"}"; path="${path#\"}"
  path="${path%"'"}"; path="${path#"'"'}"
  path="$(printf '%s' "$path" | sed 's/^[[:space:]]*//; s/[[:space:]]*$//')"

  if [[ ! -f "$path" ]]; then
    echo "\"$path\",,\"error\",\"file not found\"" >> "$OUTPUT_CSV"
    echo "✗ $path (not found)" >&2
    continue
  fi

  # Upload
  # -sS: quiet but show errors | -f: fail on 4xx/5xx | --max-time 120: avoid hanging forever
  response="$(curl -sS -f --max-time 120 -F "file=@${path}" "$API_URL" 2>&1)"
  curl_exit=$?

  if [[ $curl_exit -ne 0 ]]; then
    # curl error; log stderr message
    msg="$(printf '%s' "$response" | tr '\n' ' ' | sed 's/"/'\''/g')"
    echo "\"$path\",,\"error\",\"curl: $msg\"" >> "$OUTPUT_CSV"
    echo "✗ $path (curl error)" >&2
    continue
  fi

  # Parse URL from JSON
  if [[ $have_jq -eq 1 ]]; then
    url="$(printf '%s' "$response" | jq -r '.data.url // empty')"
  else
    # Fallback parser (best-effort if jq is unavailable)
    url="$(printf '%s' "$response" | grep -o '"url":"[^"]*"' | head -n1 | sed 's/"url":"//; s/"$//')"
    # If grep failed (e.g., different whitespace), try a tiny Python parser if available
    if [[ -z "$url" && -n "$(command -v python3 || true)" ]]; then
      url="$(python3 - <<'PY'
import json,sys
try:
    print(json.load(sys.stdin)["data"]["url"])
except Exception:
    pass
PY
      <<<"$response")"
    fi
  fi

  if [[ -n "$url" ]]; then
    echo "\"$path\",\"$url\",\"ok\",\"\"" >> "$OUTPUT_CSV"
    echo "✓ $path -> $url"
  else
    # Could not extract URL; store raw response for debugging
    msg="$(printf '%s' "$response" | tr '\n' ' ' | sed 's/"/'\''/g')"
    echo "\"$path\",,\"error\",\"no url in response: $msg\"" >> "$OUTPUT_CSV"
    echo "✗ $path (no URL in response)" >&2
  fi
done < "$INPUT_LIST"

echo
echo "Done. Results saved to: $OUTPUT_CSV"
